"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const promises_1 = __importDefault(require("node:fs/promises"));
const node_path_1 = __importDefault(require("node:path"));
class McxUtlis {
    /**
     * 检查文件是否存在
     * @param path 文件路径
     * @returns 是否存在
     */
    static async FileExsit(path) {
        try {
            await promises_1.default.access(path);
            return true;
        }
        catch {
            return false;
        }
    }
    /**
     * 读取文件内容，支持返回 string 或 object，带重试机制
     * @param filePath 文件路径
     * @param opt 配置选项
     * @returns 文件内容（string | object），出错时返回 {}
     */
    static async readFile(filePath, opt = {}) {
        // 补全必填字段，确保类型安全
        const opts = {
            delay: 200, // 默认延迟
            maxRetries: 3, // 默认最大重试次数
            want: 'string', // 默认返回字符串
            ...opt, // 用户传入的配置覆盖默认值
        };
        for (let attempt = 0; attempt < opts.maxRetries; attempt++) {
            try {
                const buffer = await promises_1.default.readFile(filePath);
                let text;
                if (opts.want === 'string') {
                    text = buffer.toString(); // Buffer -> string
                }
                else if (opts.want === 'object') {
                    try {
                        text = JSON.parse(buffer.toString()); // Buffer -> string -> object
                    }
                    catch (parseErr) {
                        // JSON 解析失败时返回空对象
                        text = {};
                    }
                }
                else {
                    // 默认情况也返回字符串
                    text = buffer.toString();
                }
                return text;
            }
            catch (err) {
                // 如果不是最后一次尝试，则等待后重试
                if (attempt < opts.maxRetries - 1) {
                    await McxUtlis.sleep(opts.delay);
                }
            }
        }
        return opts.want === 'object' ? {} : '';
    }
    static sleep(time) {
        return new Promise((resolve) => setTimeout(resolve, time));
    }
    // 在运行时进行对象类型验证
    static TypeVerify(obj, types) {
        for (const item of Object.entries(types)) {
            const [key, ShouldType] = item;
            if (!(typeof obj[key] === ShouldType))
                return false;
        }
        return true;
    }
    static AbsoluteJoin(baseDir, inputPath) {
        return node_path_1.default.isAbsolute(inputPath) ?
            inputPath :
            node_path_1.default.join(baseDir, inputPath);
    }
}
exports.default = McxUtlis;
